///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//  Copyright  NetworkDLS 2002, All rights reserved
//
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF 
// ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO 
// THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A 
// PARTICULAR PURPOSE.
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#ifndef _Common_CPP
#define _Common_CPP
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#include <Windows.H>
#include <ShlObj.H>
#include <Stdio.H>
#include <ShlOBJ.H>
#include <Stdlib.H>

#include "Debug.H"
#include "../SharedSource/NSWFL.H"
#include "Common.H"
#include "zConf.h"
#include "zLib.h"

#include "../SharedClasses/CMemPool/CMemPool.H"
#include "../SharedSource/EventLog.H"

#ifdef _SQLESERVER
	#include "../SQLEServer/Source/Routines.H"
	#include "../SQLEServer/Source/Entry.H"
	#include "../SQLEServer/Dialogs/AlertDlg.H"
#elif _SQLECLIENT
	#include "../SQLEClient/Source/Routines.H"
	#include "../SQLEClient/Source/Entry.H"
#else
	#error "You must define either _SQLESERVER or _SQLECLIENT"
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

char *sExcludedFromInit[] = {
	"/OpenConsole",
	"/OpenMainDialog",
	"/Install",
	"/Start",
	"/Delete",
	"/Stop",
	NULL
};


///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool IsApplicationRunning(void)
{
	if(IsApplicationRunningEx())
	{
		MessageBox(NULL, "The application is already running.", gsTitleCaption, MB_TASKMODAL|MB_ICONINFORMATION);
		return true;
	}
	return false;
}
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool IsApplicationRunningEx(void)
{
	bool bResult = false;
	HANDLE hMutex = NULL;
	SECURITY_DESCRIPTOR SD;
	SECURITY_ATTRIBUTES SA;

	if(InitializeSecurityDescriptor(&SD, SECURITY_DESCRIPTOR_REVISION) == FALSE)
	{
		return false;
	}

	if(SetSecurityDescriptorDacl(&SD, TRUE, 0, FALSE) == FALSE)
	{
		return false;
	}

	SA.nLength				= sizeof(SA);
	SA.lpSecurityDescriptor = &SD;
	SA.bInheritHandle		= FALSE;

	char sMutexName[1024];
	if((hMutex = CreateMutex(&SA, FALSE, GetBuildMutexName(gsTitleCaption, sizeof(sMutexName), sMutexName))) != NULL)
	{
		if(GetLastError() == ERROR_ALREADY_EXISTS)
		{
			bResult = true;
		}

		ReleaseMutex(hMutex);
		CloseHandle(hMutex);
	}

	return bResult;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool ExcludedFromInit(const char *sCmdLine)
{
	int iItem = 0;

	if(strlen(sCmdLine) == 0)
	{
		return false;
	}

	while(sExcludedFromInit[iItem])
	{
		if(_strcmpi(sExcludedFromInit[iItem], sCmdLine) == 0)
		{
			return true;
		}

		iItem++;
	}

	return false;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool DoesServerFileExist(const char *sFile)
{
/*
	CSQL MySQL;
	CRecordSet rsTemp;

	char sBuf[1024];

	int iLen = 0;

	memset(sBuf, 0, sizeof(sBuf));

	sprintf_s(sBuf, sizeof(sBuf), "XP_CmdShell 'IF EXIST \"%s\" ECHO FILE_EXIST'", sFile);

	if(MySQL.Connect(gsSQLIndexDriver, gsSQLIndexServer, gsSQLIndexUserID, gsSQLIndexPassword, "Master"))
	{
		if(MySQL.Execute(sBuf, &rsTemp))
		{
			memset(sBuf, 0, sizeof(sBuf));

			if(rsTemp.Fetch())
			{
				if(rsTemp.sColumnEx(1, sBuf, sizeof(sBuf), &iLen))
				{
					if(_strcmpi(sBuf, "FILE_EXIST") == 0)
					{
						return true;
					}
				}
			}
		}
	}
*/
	return false;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int GetListItemlParam(HWND hWnd, int iItem, int iSubItem)
{
	LV_ITEM MyLVItem;

	memset(&MyLVItem, 0, sizeof(MyLVItem));
	MyLVItem.mask       = LVIF_PARAM;
    MyLVItem.iSubItem   = iSubItem;
	MyLVItem.iItem      = iItem;

	if(ListView_GetItem(hWnd, &MyLVItem))
	{
		return MyLVItem.lParam;
	}

	return 0;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool OpenLogFile(void)
{
    char sFullPath[MAX_PATH];

    SYSTEMTIME ST;

    GetLocalTime(&ST);

    char DateTemp[64 + 1];

    GetDateFormat(LOCALE_USER_DEFAULT, NULL, &ST, "MM-dd-yy", DateTemp,64);

    sprintf_s(sFullPath, sizeof(sFullPath), "%s\\%s.txt", gsSessionLogDir, DateTemp);

	if(fopen_s(&gfhSessionLog, sFullPath, "ab") != 0)
	{
        return false;
	}

    return true;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool CloseLogFile(void)
{
	if(gfhSessionLog)
	{
		fclose(gfhSessionLog);
		return true;
	}
	else return false;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void WriteAnEvent(const char *sMessage, int iEventType)
{
	if(iEventType == EVENT_NONE)
	{
		//No need to record an event.
	}
	else if(iEventType == EVENT_INFO)
	{
		RecordInfo(sMessage);
	}
	else if(iEventType == EVENT_WARN)
	{
		RecordWarn(sMessage);
		gdwWarningCount++;
	}
	else if(iEventType == EVENT_ERROR)
	{
		RecordError(sMessage);
		gdwErrorCount++;
	}
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void WriteSysLogEx(const char *sMessage, int iEventType)
{
	if(!gbExcludeInit)
	{
		SYSTEMTIME ST;
		GetLocalTime(&ST);

		char lsDate[64];
		char lsTime[64];
		char lsTemp[MAX_STATUS_TEXT + (64 * 2) + 1];

		GetDateFormat(LOCALE_USER_DEFAULT, DATE_SHORTDATE, &ST, NULL, lsDate, 64);
		GetTimeFormat(LOCALE_USER_DEFAULT, 0, &ST, NULL, lsTime, 64);

		sprintf_s(lsTemp, sizeof(lsTemp), "(%s %s) [*] %s", lsDate, lsTime, sMessage);
		fprintf(gfhSessionLog, "%s\r\n", lsTemp);
	 	
		if(!gbCacheLogFile)
		{
			fflush(gfhSessionLog);
		}
	    
		InsertListBoxItemEx(MDI.hLogList, lsTemp, 100, -1);

		WriteAnEvent(lsTemp, iEventType);

		#ifdef _SQLESERVER
		if(iEventType == EVENT_WARN && gbVisualAlertOnWarn)
		{
			StartAlertDlg("Warning", sMessage, true);
		}

		if(iEventType == EVENT_ERROR && gbVisualAlertOnError)
		{
			StartAlertDlg("Error", sMessage, true);
		}
		#endif
	}
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void WriteSysLog(const char *sMessage)
{
	WriteSysLogEx(sMessage, EVENT_NONE);
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void WriteLogEx(int iClientId, const char *sMessage, int iEventType)
{
	if(!gbExcludeInit)
	{
		SYSTEMTIME ST;
		GetLocalTime(&ST);

		char lsDate[64];
		char lsTime[64];
		char lsTemp[MAX_STATUS_TEXT + (64 * 2) + 1];

		GetDateFormat(LOCALE_USER_DEFAULT, DATE_SHORTDATE, &ST, NULL, lsDate, 64);
		GetTimeFormat(LOCALE_USER_DEFAULT, 0, &ST, NULL, lsTime, 64);

		sprintf_s(lsTemp, sizeof(lsTemp), "(%s %s) [%d] %s", lsDate, lsTime, iClientId, sMessage);
		fprintf(gfhSessionLog, "%s\r\n", lsTemp);

		if(!gbCacheLogFile)
		{
			fflush(gfhSessionLog);
		}

		InsertListBoxItemEx(MDI.hLogList, lsTemp, 100, -1);

		WriteAnEvent(lsTemp, iEventType);

		#ifdef _SQLESERVER
		if(iEventType == EVENT_WARN && gbVisualAlertOnWarn)
		{
			StartAlertDlg("Warning", sMessage, true);
		}

		if(iEventType == EVENT_ERROR && gbVisualAlertOnError)
		{
			StartAlertDlg("Error", sMessage, true);
		}
		#endif
	}
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void WriteLog(int iClientId, const char *sMessage)
{
	WriteLogEx(iClientId, sMessage, EVENT_NONE);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int CmdCmp(const char *sBuf, const char *sCmd)
{
	int iCmdLen = strlen(sCmd);
	if( strncmp(sBuf, sCmd, iCmdLen) == 0)
		return iCmdLen;
	else return 0;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int AppendDataToCmd(const char *sCmd, char *sData, int iDataSz, char *sOutBuf)
{
	int iCmdSz = strlen(sCmd);
	int iRPos = 0;
	int iRWos = 0;

	while(iRPos < iCmdSz)
	{
		sOutBuf[iRWos++] = sCmd[iRPos++];
	}

	iRPos = 0;

	while(iRPos < iDataSz)
	{
		sOutBuf[iRWos++] = sData[iRPos++];
	}

	return iRWos;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int BreakCmdFromData(const char *sBuf, int iCmdLen, int iBufSz, char *sOutBuf)
{
	int WPos = 0;
	int RPos = iCmdLen;

	while(RPos < iBufSz)
		sOutBuf[WPos++] = sBuf[RPos++];

	sOutBuf[WPos] = '\0';

	return WPos;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void CreateFolderStructure(char *sFullPath)
{
    char Directory[MAX_PATH];

    int NameLength = strlen(sFullPath);
    int Position = 0;

    while(Position < NameLength)
    {
        if(sFullPath[Position] == '\\' || sFullPath[Position] == '/')
        {
            Directory[Position] = '\0';
            CreateDirectory(Directory, NULL);
        }

        Directory[Position] = sFullPath[Position];

        Position++;
    }
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

char *ResolveHost(const char *sHostName, int iAddress)
{
	struct in_addr Address;

	struct hostent *Host;

	memset(&Host, 0, sizeof(Host));
	memset(&Address, 0, sizeof(Address));

	if((Host = gethostbyname(sHostName)) == NULL)
	{
		return NULL;
	}

	memcpy(&Address.s_addr, (*Host).h_addr_list[iAddress], (*Host).h_length);

	return inet_ntoa(Address);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool GetIPAddress(char *sOutAddr, int iMaxSz, const char *sInputAddr)
{
	bool ValidIP = true;

    if(inet_addr(sInputAddr) == INADDR_NONE) ValidIP = false;
    if(strlen(sInputAddr) < 7)               ValidIP = false;
    if(strlen(sInputAddr) > 15)              ValidIP = false;

	if(ValidIP == false)
	{
		char *Resolved = ResolveHost(sInputAddr, 0);
		if(Resolved == NULL)
		{
			return false;
		}
		strcpy_s(sOutAddr, iMaxSz, Resolved);
	}
	else{
		strcpy_s(sOutAddr, iMaxSz, sInputAddr);
	}

	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

bool BinCompare(const void *s1, const void *s2, int iSz)
{
	int iRPos = 0;

	while(iRPos < iSz)
	{
		if(((unsigned char *)s1)[iRPos++] != ((unsigned char *)s2)[iRPos++])
		{
			return false;
		}
	}

	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

char *GetBuildMutexName(const char *sBase, int iMaxSz, char *sOutString)
{
	strcpy_s(sOutString, iMaxSz, sBase);

	#ifdef _X64_
		strcat_s(sOutString, iMaxSz, " x64");
	#elif _IA64_
		strcat_s(v, iMaxSz, " IA64");
	#else
		strcat_s(sOutString, iMaxSz, " x86");
	#endif

	return sOutString;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#endif
